<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

class DLM_DP_Access_Manager {

	private $redirect_to_download_page = false;

	/** @var DLM_Download */
	private $download = null;

	/**
	 * Setup class
	 */
	public function setup() {

		// we're preventing access to all requests that are not from the download page and with correct nonce
		add_filter( 'dlm_can_download', array( $this, 'check_access' ), 300, 3 );
		add_filter( 'dlm_download_headers', array( $this, 'dlp_headers' ), 60, 1 );
		add_filter( 'dlm_access_denied_redirect', array( $this, 'download_page_redirect' ) );
		add_action( 'wp_footer', array( $this, 'add_scripts' ) );
		add_action( 'wp_ajax_nopriv_dlm_dp_modal', array( $this, 'xhr_no_access_modal' ), 15 );
		add_action( 'wp_ajax_dlm_dp_modal', array( $this, 'xhr_no_access_modal' ), 15 );
	}

	/**
	 * Add headers to XHR request
	 *
	 * @param array $headers Headers.
	 *
	 * @return array
	 * @since 4.0.5
	 */
	public function dlp_headers( $headers ) {
		if ( $this->do_modal() ) {
			$headers['x-dlm-dlp-redirect']    = 1;
			$headers['x-dlm-no-access']       = 'true';
			$headers['x-dlm-no-access-modal'] = 'true';
		}

		return $headers;
	}

	/**
	 * Add required scripts to footer.
	 *
	 * @return void
	 */
	public function add_scripts() {
		if ( $this->do_modal() ) {
			?>
			<script>
				jQuery(document).on('dlm-xhr-modal-data', function (e, data, headers) {
					if ('undefined' != typeof headers['x-dlm-dlp-redirect']) {
						data['action'] = 'dlm_dp_modal';
						data['dlm_modal_response'] = 'true';
					}
				});
			</script>
			<?php
		}
	}

	/**
	 * Check if requester has access to download
	 *
	 * @param bool $has_access Has access.
	 * @param object $download Download object.
	 * @param object $version Version object.
	 *
	 * @return bool
	 */
	public function check_access( $has_access, $download, $version ) {

		// don't continue if access is already false.
		if ( false == $has_access ) {
			return $has_access;
		}

		// Check if XHR and modal is enabled.
		$xhr = false;
		if ( defined( 'DLM_DOING_XHR' ) && DLM_DOING_XHR ) {
			$xhr = true;
		}
		$access_modal = apply_filters( 'do_dlm_xhr_access_modal', absint( get_option( 'dlm_no_access_modal', 0 ) ), $download );

		if ( $xhr && $access_modal && $this->do_modal() ) {
			return $has_access;
		}

		// default to no access.
		$has_access                      = false;
		$this->redirect_to_download_page = true;
		$this->download                  = $download;

		// check if the force get is set and the nonce is set
		if ( ! empty( $_GET[ DLM_DP_Constants::GET_KEY_DL_FORCE ] ) && ! empty( $_GET[ DLM_DP_Constants::GET_KEY_NONCE ] ) ) {

			// verify nonce
			if ( wp_verify_nonce( $_GET[ DLM_DP_Constants::GET_KEY_NONCE ], 'dlm-dp-download-nonce' . DLM_Utils::get_visitor_ip() ) ) {

				// all checks out. Set access to true and reset internal variables to default
				$has_access                      = true;
				$this->redirect_to_download_page = false;
				$this->download                  = null;
			}
		}

		// return $has_access
		return $has_access;
	}

	/**
	 * Redirect to download page if we denied access
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function download_page_redirect( $url ) {

		// check if we're redirecting
		if ( $this->redirect_to_download_page && ! is_null( $this->download ) ) {

			// get downloading page
			$downloading_page_id = absint( get_option( DLM_DP_Constants::SETTING_PAGE, 0 ) );

			// check if a downloading page is found
			if ( $downloading_page_id > 0 ) {

				// get permalink of downloading page
				$downloading_page_permalink = get_permalink( $downloading_page_id );

				// check if we can find a permalink
				if ( false !== $downloading_page_permalink ) {

					// append download id to no access URL
					$url = add_query_arg( DLM_DP_Constants::GET_KEY_ID, $this->download->get_id(), untrailingslashit( $downloading_page_permalink ) );

					// check if download version is set
					if ( ! empty( $_GET['version'] ) ) {
						$url = add_query_arg( 'version', $_GET['version'], $url );
					}
					// SHOP - check if order_id  and order_hash are set
					if ( ! empty( $_GET['order_id'] ) && ! empty( $_GET['order_hash'] ) ) {
						$url = add_query_arg( 'order_id', $_GET['order_id'], $url );
						$url = add_query_arg( 'order_hash', $_GET['order_hash'], $url );
					}
				}

			}

		}

		return $url;
	}

	/**
	 * Generate download nonce
	 *
	 * @return string
	 */
	public static function generate_nonce() {
		return wp_create_nonce( 'dlm-dp-download-nonce' . DLM_Utils::get_visitor_ip() );
	}

	/**
	 * Generate the download URL that will actually download the file instead of redirecting to the downloading page
	 *
	 * @param DLM_Download $download
	 *
	 * @todo add nonce so the DLM_DP_Constants::GET_KEY_DL_FORCE can't be directly linked
	 *
	 * @return string
	 */
	private function generate_forced_download_url( $download ) {
		$url = add_query_arg( array(
			DLM_DP_Constants::GET_KEY_DL_FORCE => 1,
			DLM_DP_Constants::GET_KEY_NONCE    => DLM_DP_Access_Manager::generate_nonce()
		), $download->get_the_download_link() );

		if ( ! empty ( $_GET['version'] ) ) {
			$url = add_query_arg( 'version', $_GET['version'], $url );
		}

		// SHOP - check if order_id  and order_hash are set
		if ( ! empty( $_GET['order_id'] ) && ! empty( $_GET['order_hash'] ) ) {
			$url = add_query_arg( 'order_id', $_GET['order_id'], $url );
			$url = add_query_arg( 'order_hash', $_GET['order_hash'], $url );
		}
		
		return $url;
	}

	/**
	 * Log the XHR download
	 *
	 * @return void
	 */
	public function xhr_no_access_modal() {

		if ( ! isset( $_POST['download_id'] ) || ! isset( $_POST['version_id'] ) ) {
			if ( '1' === get_option( 'dlm_xsendfile_enabled' ) ) {
				wp_send_json_error( 'Missing download_id or version_id. X-Sendfile is enabled, so this is a problem.' );
			}
			wp_send_json_error( 'Missing download_id or version_id' );
		}

		check_ajax_referer( 'dlm_ajax_nonce', 'nonce' );

		// Action to allow the adition of extra scripts and code related to the shortcode.
		do_action( 'dlm_dlm_no_access_shortcode_scripts' );

		ob_start();

		// template handler.
		$template_handler = new DLM_Template_Handler();

		if ( 'empty-download' === $_POST['download_id'] || ( isset( $_POST['modal_text'] ) && ! empty( $_POST['modal_text'] ) ) ) {
			if ( isset( $_POST['modal_text'] ) && ! empty( $_POST['modal_text'] ) ) {
				echo sanitize_text_field( wp_unslash( $_POST['modal_text'] ) );
			} else {
				echo '<p>' . __( 'You do not have permission to download this file.', 'download-monitor' ) . '</p>';
			}
		} else {
			try {
				$download = download_monitor()->service( 'download_repository' )->retrieve_single( absint( $_POST['download_id'] ) );

				// Template handler.
				$template_handler      = new DLM_Template_Handler();
				$settings_helper       = new DLM_Settings_Helper();
				$download_page         = absint( $settings_helper->get_option( 'dp_downloading_page' ) );
				$download_page_title   = __( 'Downloading Page', 'dlm-downloading-page' );
				$download_page_content = '';

				if ( $download_page ) {
					$download_page_title   = get_the_title( $download_page );
					$download_page_content = get_the_content( null, false, $download_page );
				}

				// Load template.
				$template_handler->get_template_part( 'modal-template', '', plugin_dir_path( DLM_Downloading_Page::get_plugin_file() ) . 'templates/', array(
					'url'      => $this->generate_forced_download_url( $download ),
					'download' => $download,
					'content'  => do_shortcode( $download_page_content ),
					'title'    => $download_page_title,
				) );

			} catch ( Exception $exception ) {

			}
		}
		$content               = ob_get_clean();
		// Content and variables escaped above.
		// $content variable escaped from extensions as it may include inputs or other HTML elements.
		echo apply_filters( 'the_content', $content ); //phpcs:ignore

		$download = download_monitor()->service( 'download_repository' )->retrieve_single( absint( $_REQUEST['download_id'] ) );
		$url      = $this->generate_forced_download_url( $download );

		echo '<script>
		var delay = 3000; 
		setTimeout(function(){ window.location = "' . $url . '"; }, delay);
		</script>';

		die();
	}

	/**
	 * Check if we need to do the modal.
	 *
	 * @return mixed|null
	 * @since 4.0.5
	 */
	private function do_modal() {
		return apply_filters( 'dlm_dp_do_modal', false );
	}
}
