<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xolio Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_Testimonial extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'testimonial';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Testimonial', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'tp_layout',
            [
                'label' => esc_html__('Design Layout', 'tpcore'),
            ]
        );
        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'tpcore'),
                    'layout-2' => esc_html__('Layout 2', 'tpcore'),
                    'layout-3' => esc_html__('Layout 3', 'tpcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // Review group
        $this->start_controls_section(
            'review_list',
            [
                'label' => esc_html__('Review List', 'tpcore'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'review_content',
            [
                'label' => esc_html__('Review Content', 'tpcore'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => 'Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry\'s standard dummy text ever since the 1500s, when an unknown printer took a galley of type and',
                'placeholder' => esc_html__('Type your review content here', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'reviewer_image',
            [
                'label' => esc_html__('Reviewer Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'reviewer_image_size',
                'default' => 'thumbnail',
                'exclude' => [
                    'custom'
                ]
            ]
        );

        $repeater->add_control(
            'reviewer_name',
            [
                'label' => esc_html__('Reviewer Name', 'tpcore'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Davis Levin', 'tpcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'reviewer_designation',
            [
                'label' => esc_html__('Reviewer Designation', 'tpcore'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('CEO Kawasaki Inc.', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'reviews_list',
            [
                'label' => esc_html__('Review List', 'tpcore'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' =>  $repeater->get_controls(),
                'default' => [
                    [
                        'reviewer_name' => esc_html__('Davis Levin', 'tpcore'),
                    ],
                    [
                        'reviewer_name' => esc_html__('Samuel Peters', 'tpcore'),
                    ],
                    [
                        'reviewer_name' => esc_html__('Zonalos Neko', 'tpcore'),
                    ],

                ],
                'title_field' => '{{{ reviewer_name }}}',
            ]
        );

        $this->end_controls_section();


        // TAB_STYLE
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'tpcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'tpcore'),
                    'lowercase' => esc_html__('lowercase', 'tpcore'),
                    'capitalize' => esc_html__('Capitalize', 'tpcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

?>

        <?php if ($settings['tg_design_style']  == 'layout-2') : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =     Testimonial Active       =
                    =============================================*/
                    $('.testimonial-active-two').slick({
                        dots: true,
                        infinite: false,
                        speed: 1000,
                        autoplay: true,
                        arrows: false,
                        slidesToShow: 4,
                        slidesToScroll: 1,
                        responsive: [{
                                breakpoint: 1200,
                                settings: {
                                    slidesToShow: 3,
                                    slidesToScroll: 1,
                                    infinite: true,
                                }
                            },
                            {
                                breakpoint: 992,
                                settings: {
                                    slidesToShow: 2,
                                    slidesToScroll: 1
                                }
                            },
                            {
                                breakpoint: 767,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                            {
                                breakpoint: 575,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                        ]
                    });

                });
            </script>
            <div class="testimonial-area-two">
                <div class="row testimonial-active-two">
                    <?php foreach ($settings['reviews_list'] as $item) :

                        if (!empty($item['reviewer_image']['url'])) {
                            $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                            $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                        }

                    ?>
                        <div class="col-lg-3">
                            <div class="testimonial-item-two">
                                <div class="testimonial-content-two">
                                    <div class="rating">
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                    </div>
                                    <?php if (!empty($item['review_content'])) : ?>
                                        <p><?php echo esc_html($item['review_content']); ?></p>
                                    <?php endif; ?>
                                </div>
                                <div class="testimonial-info-two">
                                    <?php if (!empty($item['reviewer_image'])) : ?>
                                        <div class="thumb">
                                            <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <div class="content">
                                        <?php if (!empty($item['reviewer_name'])) : ?>
                                            <h5 class="title"><?php echo tp_kses($item['reviewer_name']); ?></h5>
                                        <?php endif; ?>

                                        <?php if (!empty($item['reviewer_designation'])) : ?>
                                            <span><?php echo tp_kses($item['reviewer_designation']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>

                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-3') : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =          Testimonial Active  =
                    =============================================*/
                    var serviceSwiper = new Swiper('.testimonial-active-three', {
                        // Optional parameters
                        loop: true,
                        slidesPerView: 1,
                        spaceBetween: 30,
                        autoplay: {
                            delay: 5000,
                            disableOnInteraction: true,
                        },
                        breakpoints: {
                            '1500': {
                                slidesPerView: 3,
                            },
                            '1200': {
                                slidesPerView: 3,
                            },
                            '992': {
                                slidesPerView: 3,
                            },
                            '768': {
                                slidesPerView: 3,
                            },
                            '576': {
                                slidesPerView: 2,
                            },
                            '0': {
                                slidesPerView: 1,
                            },
                        },
                        // Navigation arrows
                        navigation: {
                            nextEl: ".swiper-button-next",
                            prevEl: ".swiper-button-prev",
                        },

                        // If we need pagination
                        pagination: {
                            el: ".testimonial-swiper-pagination",
                            clickable: true,
                        },

                    });

                });
            </script>

            <div class="row">
                <div class="col-lg-12">
                    <div class="swiper-container testimonial-active-three">
                        <div class="swiper-wrapper">
                            <?php foreach ($settings['reviews_list'] as $item) :

                                if (!empty($item['reviewer_image']['url'])) {
                                    $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                    $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                                }

                            ?>
                                <div class="swiper-slide">
                                    <div class="testimonial-item-three">
                                        <?php if (!empty($item['reviewer_image'])) : ?>
                                            <div class="testimonial-thumb-three">
                                                <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                            </div>
                                        <?php endif; ?>
                                        <div class="testimonial-content-three">

                                            <?php if (!empty($item['reviewer_name'])) : ?>
                                                <h4 class="title"><?php echo tp_kses($item['reviewer_name']); ?></h4>
                                            <?php endif; ?>

                                            <?php if (!empty($item['reviewer_designation'])) : ?>
                                                <span><?php echo tp_kses($item['reviewer_designation']); ?></span>
                                            <?php endif; ?>

                                            <?php if (!empty($item['review_content'])) : ?>
                                                <p><?php echo esc_html($item['review_content']); ?></p>
                                            <?php endif; ?>

                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
            <div class="testimonial-swiper-pagination"></div>

        <?php else : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =       Testimonial Active      =
                    =============================================*/
                    $('.testimonial-active').slick({
                        dots: true,
                        infinite: false,
                        speed: 1000,
                        autoplay: true,
                        arrows: false,
                        slidesToShow: 4,
                        slidesToScroll: 1,
                        responsive: [{
                                breakpoint: 1200,
                                settings: {
                                    slidesToShow: 3,
                                    slidesToScroll: 1,
                                    infinite: true,
                                }
                            },
                            {
                                breakpoint: 992,
                                settings: {
                                    slidesToShow: 2,
                                    slidesToScroll: 1
                                }
                            },
                            {
                                breakpoint: 767,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                            {
                                breakpoint: 575,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                        ]
                    });

                });
            </script>

            <div class="testimonial-item-wrap">
                <div class="row testimonial-active">
                    <?php foreach ($settings['reviews_list'] as $item) :

                        if (!empty($item['reviewer_image']['url'])) {
                            $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                            $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                        }

                    ?>
                        <div class="col-lg-3">
                            <div class="testimonial-item">
                                <?php if (!empty($item['review_content'])) : ?>
                                    <div class="testimonial-content">
                                        <p><?php echo esc_html($item['review_content']); ?></p>
                                    </div>
                                <?php endif; ?>
                                <div class="testimonial-info">
                                    <?php if (!empty($item['reviewer_image'])) : ?>
                                        <div class="thumb">
                                            <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <div class="content">
                                        <?php if (!empty($item['reviewer_name'])) : ?>
                                            <h4 class="title"><?php echo tp_kses($item['reviewer_name']); ?></h4>
                                        <?php endif; ?>

                                        <?php if (!empty($item['reviewer_designation'])) : ?>
                                            <p><?php echo tp_kses($item['reviewer_designation']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_Testimonial());
