<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xolio Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_IconBox extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tg-iconbox';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Icon Box', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'tg_layout',
            [
                'label' => esc_html__('Design Layout', 'tpcore'),
            ]
        );
        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'tpcore'),
                    'layout-2' => esc_html__('Layout 2', 'tpcore'),
                    'layout-3' => esc_html__('Layout 3', 'tpcore'),
                    'layout-4' => esc_html__('Layout 4', 'tpcore'),
                    'layout-5' => esc_html__('Layout 5', 'tpcore'),
                    'layout-6' => esc_html__('Layout 6', 'tpcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // Style_group
        $this->start_controls_section(
            'tg_iconbox_group',
            [
                'label' => esc_html__('IconBox Group', 'tpcore'),
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'item_animation_delay',
            [
                'label' => esc_html__('Animation Delay(.ms)', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('.2', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tg_icon_type',
            [
                'label' => esc_html__('Select Icon Type', 'tpcore'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'image' => esc_html__('Image', 'tpcore'),
                    'icon' => esc_html__('Icon', 'tpcore'),
                ],
            ]
        );

        $repeater->add_control(
            'tg_svg',
            [
                'label' => esc_html__('Upload Icon', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'tg_icon_type' => 'image'
                ]
            ]
        );

        if (tp_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'tg_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                    'condition' => [
                        'tg_icon_type' => 'icon'
                    ]
                ]
            );
        } else {
            $repeater->add_control(
                'tg_selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'fas fa-star',
                        'library' => 'solid',
                    ],
                    'condition' => [
                        'tg_icon_type' => 'icon'
                    ]
                ]
            );
        }

        $repeater->add_control(
            'tg_title',
            [
                'label' => esc_html__('Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('UI UX Design', 'tpcore'),
                'placeholder' => esc_html__('Type Heading Text', 'tpcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tg_description',
            [
                'label' => esc_html__('Description', 'tpcore'),
                'description' => tp_get_allowed_html_desc('intermediate'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Lorem Ipsum is simply dummy text of the printing and typesetting industry.', 'tpcore'),
                'placeholder' => esc_html__('Type section description here', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tg_btn_text',
            [
                'label' => esc_html__('Button Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'tpcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tg_btn_link_type',
            [
                'label' => esc_html__('Item Link Type', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tg_btn_link',
            [
                'label' => esc_html__('Item link', 'tpcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'tpcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'tg_btn_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tg_btn_page_link',
            [
                'label' => esc_html__('Select Item Page', 'tpcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => tp_get_all_pages(),
                'condition' => [
                    'tg_btn_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
            'item_lists',
            [
                'label' => esc_html__('Item Box Lists', 'tpcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'tg_title' => esc_html__('UI UX Design', 'tpcore'),
                    ],
                    [
                        'tg_title' => esc_html__('Product Design', 'tpcore'),
                    ],
                    [
                        'tg_title' => esc_html__('Website Design', 'tpcore'),
                    ],
                ],
            ]
        );

        $this->end_controls_section();


        // Style TAB
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'tpcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'tpcore'),
                    'lowercase' => esc_html__('lowercase', 'tpcore'),
                    'capitalize' => esc_html__('Capitalize', 'tpcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

?>

        <?php if ($settings['tg_design_style']  == 'layout-2') : ?>

            <div class="row justify-content-center">
                <?php foreach ($settings['item_lists'] as $key => $item) :

                    $is_active = ($key == '0') ? "active" : "";

                    // Link
                    if ('2' == $item['tg_btn_link_type']) {
                        $link = get_permalink($item['tg_btn_page_link']);
                        $target = '_self';
                        $rel = 'nofollow';
                    } else {
                        $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                        $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                        $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                    }

                ?>
                    <div class="col-xl-3 col-lg-4 col-md-6 col-sm-8">
                        <div class="services-item-two <?php echo esc_attr($is_active) ?>">
                            <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                    <div class="services-icon-two">
                                        <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                    </div>
                                <?php endif; ?>
                            <?php else : ?>
                                <?php if (!empty($item['tg_svg']['url'])) : ?>
                                    <div class="services-icon-two">
                                        <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                            <div class="services-content-two">
                                <h4 class="title">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                        <?php echo tp_kses($item['tg_title']); ?>
                                    </a>
                                </h4>
                                <?php if (!empty($item['tg_description'])) : ?>
                                    <p><?php echo tp_kses($item['tg_description']); ?></p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>

            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-3') : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =       Services Active   =
                    =============================================*/
                    $('.services-active').slick({
                        dots: true,
                        infinite: false,
                        speed: 1000,
                        autoplay: true,
                        arrows: false,
                        slidesToShow: 3,
                        slidesToScroll: 1,
                        responsive: [{
                                breakpoint: 1200,
                                settings: {
                                    slidesToShow: 3,
                                    slidesToScroll: 1,
                                    infinite: true,
                                }
                            },
                            {
                                breakpoint: 992,
                                settings: {
                                    slidesToShow: 2,
                                    slidesToScroll: 1
                                }
                            },
                            {
                                breakpoint: 767,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                            {
                                breakpoint: 575,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                        ]
                    });

                });
            </script>

            <div class="row services-active">
                <?php foreach ($settings['item_lists'] as $item) :
                    // Link
                    if ('2' == $item['tg_btn_link_type']) {
                        $link = get_permalink($item['tg_btn_page_link']);
                        $target = '_self';
                        $rel = 'nofollow';
                    } else {
                        $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                        $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                        $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                    }
                ?>
                    <div class="col-lg-4">
                        <div class="services-item-three">
                            <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                    <div class="services-icon-three">
                                        <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                    </div>
                                <?php endif; ?>
                            <?php else : ?>
                                <?php if (!empty($item['tg_svg']['url'])) : ?>
                                    <div class="services-icon-three">
                                        <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                            <div class="services-content-three">
                                <h4 class="title">
                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                        <?php echo tp_kses($item['tg_title']); ?>
                                    </a>
                                </h4>
                                <?php if (!empty($item['tg_description'])) : ?>
                                    <p><?php echo tp_kses($item['tg_description']); ?></p>
                                <?php endif; ?>

                                <?php if (!empty($item['tg_btn_text'])) : ?>
                                    <a href="<?php echo esc_url($link); ?>" class="link-btn"><?php echo esc_html($item['tg_btn_text']) ?> <i class="fal fa-long-arrow-right"></i></a>
                                <?php endif; ?>

                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-4') : ?>

            <div class="inner-services-item-wrap">
                <div class="row justify-content-center">
                    <?php foreach ($settings['item_lists'] as $item) :

                        // Link
                        if ('2' == $item['tg_btn_link_type']) {
                            $link = get_permalink($item['tg_btn_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                            $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                        }

                    ?>
                        <div class="col-xl-3 col-lg-4 col-md-6 col-sm-8">
                            <div class="services-item-four">
                                <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                    <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                        <div class="services-icon-four">
                                            <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                        </div>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <?php if (!empty($item['tg_svg']['url'])) : ?>
                                        <div class="services-icon-four">
                                            <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                                <div class="services-content-four">
                                    <h4 class="title">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php echo tp_kses($item['tg_title']); ?>
                                        </a>
                                    </h4>
                                    <?php if (!empty($item['tg_description'])) : ?>
                                        <p><?php echo tp_kses($item['tg_description']); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-5') : ?>

            <div class="inner-services-wrap-two">
                <div class="row justify-content-center">
                    <?php foreach ($settings['item_lists'] as $item) :

                        // Link
                        if ('2' == $item['tg_btn_link_type']) {
                            $link = get_permalink($item['tg_btn_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                            $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                        }

                    ?>
                        <div class="col-xl-3 col-lg-3 col-md-6 col-sm-10">
                            <div class="services-item-five">
                                <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                    <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                        <div class="services-icon-five">
                                            <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                        </div>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <?php if (!empty($item['tg_svg']['url'])) : ?>
                                        <div class="services-icon-five">
                                            <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                                <div class="services-content-five">
                                    <h4 class="title">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php echo tp_kses($item['tg_title']); ?>
                                        </a>
                                    </h4>
                                    <?php if (!empty($item['tg_description'])) : ?>
                                        <p><?php echo tp_kses($item['tg_description']); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-6') : ?>

            <div class="tools-item-wrap">
                <div class="row justify-content-center">
                    <?php foreach ($settings['item_lists'] as $item) :

                        // Link
                        if ('2' == $item['tg_btn_link_type']) {
                            $link = get_permalink($item['tg_btn_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                            $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                        }

                    ?>
                        <div class="col-xl-3 col-lg-4 col-md-6 col-sm-8">
                            <div class="tools-item">
                                <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                    <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                        <div class="tools-icon">
                                            <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                        </div>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <?php if (!empty($item['tg_svg']['url'])) : ?>
                                        <div class="tools-icon">
                                            <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                                <div class="tools-content">
                                    <h4 class="title">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php echo tp_kses($item['tg_title']); ?>
                                        </a>
                                    </h4>
                                    <?php if (!empty($item['tg_description'])) : ?>
                                        <p><?php echo tp_kses($item['tg_description']); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

        <?php else : ?>

            <section class="services-area">
                <div class="container">
                    <div class="row justify-content-center">
                        <?php foreach ($settings['item_lists'] as $item) :

                            // Link
                            if ('2' == $item['tg_btn_link_type']) {
                                $link = get_permalink($item['tg_btn_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['tg_btn_link']['url']) ? $item['tg_btn_link']['url'] : '';
                                $target = !empty($item['tg_btn_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['tg_btn_link']['nofollow']) ? 'nofollow' : '';
                            }

                        ?>
                            <div class="col-xl-3 col-lg-4 col-md-6 col-sm-10">
                                <div class="services-item wow fadeInUp" data-wow-delay="<?php echo esc_attr($item['item_animation_delay']) ?>s">
                                    <?php if ($item['tg_icon_type'] !== 'image') : ?>
                                        <?php if (!empty($item['tg_icon']) || !empty($item['tg_selected_icon']['value'])) : ?>
                                            <div class="services-icon">
                                                <?php tp_render_icon($item, 'tg_icon', 'tg_selected_icon'); ?>
                                            </div>
                                        <?php endif; ?>
                                    <?php else : ?>
                                        <?php if (!empty($item['tg_svg']['url'])) : ?>
                                            <div class="services-icon">
                                                <img src="<?php echo esc_url($item['tg_svg']['url']); ?>" alt="<?php echo esc_html__('Icon', 'tpcore') ?>">
                                            </div>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                    <div class="services-content">
                                        <h4 class="title">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo tp_kses($item['tg_title']); ?>
                                            </a>
                                        </h4>
                                        <?php if (!empty($item['tg_description'])) : ?>
                                            <p><?php echo tp_kses($item['tg_description']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_IconBox());
