<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Background;
use \Elementor\Control_Media;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xolio Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_Hero_Banner extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'hero-banner';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Hero Banner', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // Slider
        $this->start_controls_section(
            'tg_slider_area',
            [
                'label' => esc_html__('Slider Area', 'tpcore'),
            ]
        );

        $slider = new \Elementor\Repeater();

        $slider->add_control(
            'slider_img',
            [
                'label' => esc_html__('Choose Image', 'tpcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $slider->add_control(
            'slider_title',
            [
                'label' => esc_html__('Title', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Digital Agency Business Make Big Deal.', 'tpcore'),
                'label_block' => true,
            ]
        );

        $slider->add_control(
            'slider_desc',
            [
                'label' => esc_html__('Description', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the', 'tpcore'),
                'label_block' => true,
            ]
        );

        $slider->add_control(
            'slider_button_show',
            [
                'label' => esc_html__('Show Button', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $slider->add_control(
            'slider_btn_text',
            [
                'label' => esc_html__('Button Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Get Started', 'tpcore'),
                'title' => esc_html__('Enter button text', 'tpcore'),
                'label_block' => true,
                'condition' => [
                    'slider_button_show' => 'yes'
                ],
            ]
        );

        $slider->add_control(
            'slider_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'slider_button_show' => 'yes'
                ],
            ]
        );

        $slider->add_control(
            'slider_btn_link',
            [
                'label' => esc_html__('Button link', 'tpcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'tpcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'slider_btn_link_type' => '1',
                    'slider_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $slider->add_control(
            'slider_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'tpcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => tp_get_all_pages(),
                'condition' => [
                    'slider_btn_link_type' => '2',
                    'slider_button_show' => 'yes'
                ]
            ]
        );

        $slider->add_control(
            'slider_video_button',
            [
                'label' => esc_html__('Show Video Button', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $slider->add_control(
            'video_btn_text',
            [
                'label' => esc_html__('Video Button Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('See Live Demo', 'tpcore'),
                'title' => esc_html__('Enter button text', 'tpcore'),
                'label_block' => true,
                'condition' => [
                    'slider_video_button' => 'yes'
                ],
            ]
        );

        $slider->add_control(
            'video_btn_link',
            [
                'label' => esc_html__('Video Button link', 'tpcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'tpcore'),
                'default' => [
                    'url' => 'https://www.youtube.com/watch?v=bixR-KIJKYM',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'slider_video_button' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'slider_info_lists',
            [
                'label' => esc_html__('Slider Lists', 'tpcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $slider->get_controls(),
                'default' => [
                    [
                        'slider_sub_title' => esc_html__('Digital Agency Business Make Big Deal.', 'tpcore'),
                    ],
                    [
                        'slider_sub_title' => esc_html__('Digital Agency Business Make Big Deal.', 'tpcore'),
                    ],
                    [
                        'slider_sub_title' => esc_html__('Digital Agency Business Make Big Deal.', 'tpcore'),
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'exclude' => ['custom'],
                'default' => 'full',
            ]
        );

        $this->add_control(
            'show_objects',
            [
                'label' => esc_html__('Show Background Objects', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();


        // TAB_STYLE
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'tpcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'tpcore'),
                    'lowercase' => esc_html__('lowercase', 'tpcore'),
                    'capitalize' => esc_html__('Capitalize', 'tpcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }


    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display(); ?>


        <!-- slider-area -->
        <section class="slider-area">
            <div class="slider-active">
                <?php foreach ($settings['slider_info_lists'] as $item) :

                    if (!empty($item['slider_img']['url'])) {
                        $tg_slider_image_url = !empty($item['slider_img']['id']) ? wp_get_attachment_image_url($item['slider_img']['id'], $settings['thumbnail_size']) : $item['slider_img']['url'];
                        $tg_slider_image_alt = get_post_meta($item["slider_img"]["id"], "_wp_attachment_image_alt", true);
                    }

                    // btn Link 01
                    if ('2' == $item['slider_btn_link_type']) {
                        $link = get_permalink($item['slider_btn_page_link']);
                        $target = '_self';
                        $rel = 'nofollow';
                    } else {
                        $link = !empty($item['slider_btn_link']['url']) ? $item['slider_btn_link']['url'] : '';
                        $target = !empty($item['slider_btn_link']['is_external']) ? '_blank' : '';
                        $rel = !empty($item['slider_btn_link']['nofollow']) ? 'nofollow' : '';
                    }

                ?>
                    <div class="single-slider">
                        <div class="container">
                            <div class="row align-items-center">

                                <?php if (!empty($tg_slider_image_url)) : ?>
                                    <div class="col-lg-6 order-0 order-lg-2">
                                        <div class="slider-img text-end" data-animation="fadeInRight" data-delay=".8s">
                                            <img src="<?php echo esc_url($tg_slider_image_url); ?>" alt="<?php echo esc_attr($tg_slider_image_alt) ?>">
                                        </div>
                                    </div>
                                <?php endif;

                                $col_12 = $tg_slider_image_url ? '6' : '12';

                                ?>

                                <div class="col-lg-<?php echo esc_attr($col_12); ?>">
                                    <div class="slider-content">

                                        <?php if (!empty($item['slider_title'])) : ?>
                                            <h2 class="title" data-animation="fadeInUp" data-delay=".2s"><?php echo tp_kses($item['slider_title']) ?></h2>
                                        <?php endif; ?>

                                        <?php if (!empty($item['slider_desc'])) : ?>
                                            <p data-animation="fadeInUp" data-delay=".4s"><?php echo tp_kses($item['slider_desc']) ?></p>
                                        <?php endif; ?>

                                        <?php if (!empty($item['slider_button_show'] || $item['slider_video_button'])) : ?>
                                            <div class="slider-btn">

                                                <?php if (!empty($item['slider_button_show'])) : ?>
                                                    <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>" class="btn" data-animation="fadeInLeft" data-delay=".6s"><?php echo esc_html($item['slider_btn_text']) ?> <span></span></a>
                                                <?php endif; ?>

                                                <?php if (!empty($item['slider_button_show'])) : ?>
                                                    <a href="<?php echo esc_url($item['video_btn_link']['url']); ?>" class="popup-video" data-animation="fadeInRight" data-delay=".6s"><?php echo esc_html($item['video_btn_text']) ?> <i class="fas fa-play pulse"></i></a>
                                                <?php endif; ?>

                                            </div>
                                        <?php endif; ?>

                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php if (!empty($settings['show_objects'])) : ?>
                <div class="slider-shape-wrap">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape01.png" alt="img">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape02.png" alt="img">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape03.png" alt="img">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape04.png" alt="img">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape05.png" alt="img">
                    <img src="<?php echo get_template_directory_uri(); ?>/assets/img/slider/slider_shape06.png" alt="img">
                </div>
            <?php endif; ?>

        </section>
        <!-- slider-area-end -->

<?php
    }
}

$widgets_manager->register(new TG_Hero_Banner());
